/*
 * sound/soc/i1sevk.c
 *
 * Author: Cao Rongrong <rrcao@ambarella.com>
 *
 * History:
 *	2011/03/28 - [Cao Rongrong] Created file
 *
 * Copyright (C) 2004-2009, Ambarella, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/io.h>
#include <linux/platform_device.h>
#include <sound/soc.h>
#include <sound/jack.h>
#include <asm/mach-types.h>
#include <mach/gpio.h>
#include <plat/audio.h>

#include "amdroid_jack.h"
#include "ambarella_i2s.h"

#include "../codecs/tlv320aic326x.h"

/* If don't use alsa standard Jack detction mechanism,
 * switch class will take over this job. */
static unsigned int gpio_jack = 1;
module_param(gpio_jack, uint, S_IRUGO);
MODULE_PARM_DESC(gpio_jack, "Whether or not use alsa standard Jack detection mechanism.");

/* Headset jack */
static struct snd_soc_jack hs_jack;

/* Headset jack detection DAPM pins */
static struct snd_soc_jack_pin hs_jack_pins[] = {
	{
		.pin = "Headset Mic",
		.mask = SND_JACK_MICROPHONE,
	},
	{
		.pin = "Headset Stereophone",
		.mask = SND_JACK_HEADPHONE,
	},
};

/* Headset jack detection gpios */
static struct snd_soc_jack_gpio hs_jack_gpios[] = {
	{
		.gpio = GPIO(12),
		.name = "hsdet-gpio",
		.report = SND_JACK_HEADSET,
		.debounce_time = 200,
	},
};

/* AV jack */
static struct snd_soc_jack av_jack;

/* AV jack detection DAPM pins */
static struct snd_soc_jack_pin av_jack_pins[] = {
	{
		.pin = "Line Out 2",
		.mask = SND_JACK_LINEOUT,
	},
};

/* AV jack detection gpios */
static struct snd_soc_jack_gpio av_jack_gpios[] = {
	{
		.gpio = GPIO(13),
		.name = "avdet-gpio",
		.report = SND_JACK_LINEOUT,
		.debounce_time = 200,
	},
};

static struct amdroid_jack_zone amdroid_jack_zones[] = {
	{
		/* 0 <= adc <= 0x200, unstable zone, default to 3pole if it stays
		 * in this range for a half second (20ms delays, 25 samples) */
		.adc_high = 0x200,
		.delay_ms = 20,
		.check_count = 25,
		.jack_type = AMDROID_HEADSET_3POLE,
	},
	{
		/* 200 < adc, 4 pole zone, default to 4pole if it stays
		 * in this range for half second (20ms delays, 10 samples) */
		.adc_high = 0x7fffffff,
		.delay_ms = 20,
		.check_count = 25,
		.jack_type = AMDROID_HEADSET_4POLE,
	},
};

static void platform_amdroid_jack_release(struct device * dev)
{
	return ;
}

static void jack_set_micbias_state(void *private_data, bool on)
{
	struct snd_soc_dapm_context *dapm = private_data;

	if (on)
		snd_soc_dapm_force_enable_pin(dapm, "MICBIAS2");
	else
		snd_soc_dapm_disable_pin(dapm, "MICBIAS2");

	snd_soc_dapm_sync(dapm);
}

struct amdroid_jack_platform_data amdroid_jack_data = {
	.set_micbias_state = jack_set_micbias_state,
	.adc_channel = 0,
	.zones = amdroid_jack_zones,
	.num_zones = ARRAY_SIZE(amdroid_jack_zones),
	.detect_gpio = GPIO(12),
	.active_high = 1,
};

static struct platform_device amdroid_jack_device = {
       .name             = "amdroid_jack",
       .dev = {
               .platform_data = &amdroid_jack_data,
	       .release	= platform_amdroid_jack_release,
       }
};

static const struct snd_soc_dapm_widget ppga3_dapm_widgets[] = {
	/* Output */
//	SND_SOC_DAPM_SPK("Ext Left Spk", NULL),
	SND_SOC_DAPM_SPK("Ext Right Spk", NULL),
//	SND_SOC_DAPM_SPK("Earphone", NULL),
	SND_SOC_DAPM_HP("Headset Stereophone", NULL),
	SND_SOC_DAPM_LINE("Line Out 1", NULL),
//	SND_SOC_DAPM_LINE("Line Out 2", NULL),
	/* Input */
	SND_SOC_DAPM_MIC("Main Mic", NULL),
	SND_SOC_DAPM_MIC("2nd Mic", NULL),
	SND_SOC_DAPM_MIC("Headset Mic", NULL),
//	SND_SOC_DAPM_LINE("FM Left In", NULL),
//	SND_SOC_DAPM_LINE("FM Right In", NULL),
//	SND_SOC_DAPM_LINE("3G In", NULL),
};

static const struct snd_soc_dapm_route ppga3_dapm_routes[] = {
//	{"Ext Left Spk", NULL, "SPKOUTLP"},
//	{"Ext Left Spk", NULL, "SPKOUTLN"},

	{"Ext Right Spk", NULL, "SPKOUTRP"},
	{"Ext Right Spk", NULL, "SPKOUTRN"},

//	{"Earphone", NULL, "HPOUT2N"},
//	{"Earphone", NULL, "HPOUT2P"},

	{"Headset Stereophone", NULL, "HPOUT1L"},
	{"Headset Stereophone", NULL, "HPOUT1R"},

	{"Line Out 1", NULL, "LINEOUT1N"},
	{"Line Out 1", NULL, "LINEOUT1P"},

//	{"Line Out 2", NULL, "LINEOUT2N"},
//	{"Line Out 2", NULL, "LINEOUT2P"},

	{"IN1LN", NULL, "MICBIAS1"},
	{"MICBIAS1", NULL, "Main Mic"},

//	{"IN1LP", NULL, "FM Left In"},

	{"IN2LN", NULL, "MICBIAS2"},
	{"MICBIAS2", NULL, "Headset Mic"},

	{"IN1RN", NULL, "MICBIAS1"},
	{"MICBIAS1", NULL, "2nd Mic"},

//	{"IN1RP", NULL, "FM Right In"},

//	{"IN2LP:VXRN", NULL, "3G In"},
//	{"IN2RP:VXRP", NULL, "3G In"},

};

static int ppga3_aic326x_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context *dapm = &codec->dapm;
	int errorCode = 0;

	//printk(KERN_ALERT "ppga3: ppga3_aic326x_init\r\n");
	/* add ppga3 specific widgets */
	snd_soc_dapm_new_controls(dapm, ppga3_dapm_widgets,
			ARRAY_SIZE(ppga3_dapm_widgets));

	/* set up ppga3 specific audio routes */
	snd_soc_dapm_add_routes(dapm, ppga3_dapm_routes,
			ARRAY_SIZE(ppga3_dapm_routes));

	snd_soc_dapm_sync(dapm);

	ambarella_i2s_add_controls(codec);

	if (gpio_jack) {
		/* Headset jack detection */
		errorCode = snd_soc_jack_new(codec,
				"Headset Jack", SND_JACK_HEADSET, &hs_jack);
		if (errorCode)
			return errorCode;

		errorCode = snd_soc_jack_add_pins(&hs_jack,
				ARRAY_SIZE(hs_jack_pins), hs_jack_pins);
		if (errorCode)
			return errorCode;

		errorCode = snd_soc_jack_add_gpios(&hs_jack,
				ARRAY_SIZE(hs_jack_gpios), hs_jack_gpios);
		if (errorCode)
			return errorCode;

		/* Headset jack detection */
		errorCode = snd_soc_jack_new(codec,
				"AV Jack", SND_JACK_LINEOUT, &av_jack);
		if (errorCode)
			return errorCode;

		errorCode = snd_soc_jack_add_pins(&av_jack,
				ARRAY_SIZE(av_jack_pins), av_jack_pins);
		if (errorCode)
			return errorCode;

		errorCode = snd_soc_jack_add_gpios(&av_jack,
				ARRAY_SIZE(av_jack_gpios), av_jack_gpios);
	} else {
		amdroid_jack_data.private_data = dapm;
		platform_device_register(&amdroid_jack_device);
		printk("ppga3_aic326x_init: no gpio_jack\n");
	}

	return errorCode;
}

static int ppga3_hifi_hw_params(struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	int errorCode = 0, amb_mclk, mclk, oversample;
//	printk(KERN_ALERT "ppga3: ppga3_hifi_hw_params\r\n");
	switch (params_rate(params)) {
	case 8000:
		amb_mclk = AudioCodec_4_096M;
		mclk = 4096000;
		oversample = AudioCodec_512xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 11025:
		amb_mclk = AudioCodec_5_6448M;
		mclk = 5644800;
		oversample = AudioCodec_512xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 16000:
		amb_mclk = AudioCodec_4_096M;
		mclk = 4096000;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 22050:
		amb_mclk = AudioCodec_5_6448M;
		mclk = 5644800;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 24000:
		amb_mclk = AudioCodec_6_144;
		mclk = 6144000;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 32000:
		amb_mclk = AudioCodec_8_192M;
		mclk = 8192000;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 44100:
		amb_mclk = AudioCodec_11_2896M;
		mclk = 11289600;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	case 48000:
		amb_mclk = AudioCodec_12_288M;
		mclk = 12288000;
		oversample = AudioCodec_256xfs;
		printk("ppga3: mclk = %d\n", mclk);
		break;
	default:
		errorCode = -EINVAL;
		goto hifi_hw_params_exit;
	}

	/* set the I2S system data format*/
	errorCode = snd_soc_dai_set_fmt(codec_dai,
		SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBS_CFS);
	if (errorCode < 0) {
		pr_err("can't set codec DAI configuration\n");
		goto hifi_hw_params_exit;
	}

	errorCode = snd_soc_dai_set_fmt(cpu_dai,
		SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBS_CFS);
	if (errorCode < 0) {
		pr_err("can't set cpu DAI configuration\n");
		goto hifi_hw_params_exit;
	}

	/* set the I2S system clock*/
	errorCode = snd_soc_dai_set_sysclk(cpu_dai, AMBARELLA_CLKSRC_ONCHIP, amb_mclk, 0);
	if (errorCode < 0) {
		pr_err("can't set cpu MCLK configuration\n");
		goto hifi_hw_params_exit;
	}

	errorCode = snd_soc_dai_set_clkdiv(cpu_dai, AMBARELLA_CLKDIV_LRCLK, oversample);
	if (errorCode < 0) {
		pr_err("can't set cpu MCLK/SF ratio\n");
		goto hifi_hw_params_exit;
	}

	errorCode = snd_soc_dai_set_sysclk(codec_dai, 0, mclk, 0);
	if (errorCode < 0) {
		pr_err("can't set cpu MCLK configuration\n");
		printk("can't set cpu MCLK configuration\n");
		goto hifi_hw_params_exit;
	}

hifi_hw_params_exit:
	return errorCode;
}

static struct snd_soc_ops ppga3_hifi_ops = {
	.hw_params = ppga3_hifi_hw_params,
};

static int ppga3_voice_hw_params(struct snd_pcm_substream *substream,
		struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	int errorCode = 0;

	if (params_rate(params) != 8000) {
		pr_err("Voice dai only support 8000Hz sample rate!\n");
		return -EINVAL;
	}

	errorCode = snd_soc_dai_set_fmt(codec_dai,
		SND_SOC_DAIFMT_DSP_B | SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBS_CFS);
	if (errorCode < 0) {
		pr_err("can't set codec pll configuration\n");
		goto voice_hw_params_exit;
	}

#if 0	/* When enable these codes, we need to specify BCLK2_FREQ. */
	/* Set FLL2, use BCLK2 as its source */
	errorCode = snd_soc_dai_set_pll(codec_dai, WM8994_FLL2, WM8994_FLL_SRC_BCLK,
					BCLK2_FREQ, params_rate(params) * 512);
	if (errorCode < 0) {
		pr_err("can't set codec FLL2 configuration\n");
		goto voice_hw_params_exit;
	}

	errorCode = snd_soc_dai_set_sysclk(codec_dai, WM8994_SYSCLK_FLL2,
					params_rate(params) * 512, SND_SOC_CLOCK_IN);
	if (errorCode < 0) {
		pr_err("can't set codec SYSCLK configuration\n");
		goto voice_hw_params_exit;
	}
#endif
	return 0;

voice_hw_params_exit:
	return errorCode;
}

static struct snd_soc_ops ppga3_voice_ops = {
	.hw_params = ppga3_voice_hw_params,
};

static struct snd_soc_dai_link ppga3_dai_link[] = {
	{
		.name = "AIC326X",
		.stream_name = "AIC326X-STREAM",
		.cpu_dai_name = "ambarella-i2s.0",
		.codec_dai_name = "aic326x",
		.platform_name = "ambarella-pcm-audio",
		.codec_name = "aic326x-codec.0-0018",
		.init = ppga3_aic326x_init,
		.ops = &ppga3_hifi_ops,
	},
};

static struct snd_soc_card snd_soc_card_ppga3 = {
	.name = "ppga3",
	.dai_link = ppga3_dai_link,
	.num_links = ARRAY_SIZE(ppga3_dai_link),
};


static struct platform_device *ppga3_snd_device;

static int __init ppga3_board_init(void)
{
	int errorCode = 0;
//	printk(KERN_ALERT "ppga3: ppga3_board_init start\r\n");
	ppga3_snd_device = platform_device_alloc("soc-audio", -1);
	if (!ppga3_snd_device)
		return -ENOMEM;

	platform_set_drvdata(ppga3_snd_device, &snd_soc_card_ppga3);

	errorCode = platform_device_add(ppga3_snd_device);
	if (errorCode)
		goto ppga3_board_init_exit;
//	printk(KERN_ALERT "ppga3: ppga3_board_init over\r\n");
	return 0;

ppga3_board_init_exit:
	platform_device_put(ppga3_snd_device);
	return errorCode;
}

static void __exit ppga3_board_exit(void)
{
	if (gpio_jack) {
		snd_soc_jack_free_gpios(&hs_jack,
			ARRAY_SIZE(hs_jack_gpios), hs_jack_gpios);
		snd_soc_jack_free_gpios(&av_jack,
			ARRAY_SIZE(av_jack_gpios), av_jack_gpios);
	} else {
		platform_device_unregister(&amdroid_jack_device);
	}

	platform_device_unregister(ppga3_snd_device);
}

module_init(ppga3_board_init);
module_exit(ppga3_board_exit);

MODULE_AUTHOR("Sherry Chung");
MODULE_DESCRIPTION("Amabrella ppga3 Board with TLV320AIC3262 Codec for ALSA");
MODULE_LICENSE("GPL");
