/**
 * system/src/peripheral/rct_audio.c
 *
 * History:
 *    2005/07/25 - [Charles Chiou] created file
 *    2008/02/19 - [Allen Wang] changed to use capabilities and chip ID
 *
 * Copyright (C) 2004-2008, Ambarella, Inc.
 *
 * All rights reserved. No Part of this file may be reproduced, stored
 * in a retrieval system, or transmitted, in any form, or by any means,
 * electronic, mechanical, photocopying, recording, or otherwise,
 * without the prior consent of Ambarella, Inc.
 */

#if (RCT_AUDIO_PLL_USE_HAL_API == 1)

u32 get_audio_freq_hz(void)
{
	return  (u32) amb_get_audio_clock_frequency(HAL_BASE_VP);
}
#else
extern void rct_alan_zhu_magic_loop(int clk_chk);
extern u32 get_spclk_freq_hz(void);

static const u8 audiopll_ref_table[3][2] = {
	{0x00, 0x00}, {0x01, 0x00}, {0x01, 0x01}
};
static u32 aud_sysclk = 12288000;

static void rct_set_audio_freq_hz(u32 aud_clk)
{
	aud_sysclk = aud_clk;
}

u32 get_audio_freq_hz(void)
{
	return aud_sysclk;
}


#endif

void rct_set_pll_frac_mode(void)
{
#if (RCT_AUDIO_PLL_CONF_MODE == 2)
	/* set audio PLL ctrl3 register */
	writel(PLL_AUDIO_CTRL3_REG, 0x00069300); //need set this register to enable fractional mode by a5/a2s/a2m pm
#endif
}

void rct_set_aud_ctrl2_reg(void)
{
#if (RCT_AUDIO_PLL_CONF_MODE == 2)
	/* set audio PLL ctrl2 register */
	writel(PLL_AUDIO_CTRL2_REG, 0x3f770000); //set as program manual comment
#endif
}

static const u32 mclk_table[MAX_MCLK_IDX_NUM] = {
	18432000,
	16934400,
	12288000,
	11289600,
	  9216000,
	  8467200,
	  8192000,
	  6144000,
	  5644800,
	  4608000,
	  4233600,
	  4096000,
	  3072000,
	  2822400,
	  2048000
};

#if (RCT_AUDIO_PLL_USE_HAL_API == 1)
void rct_set_audio_pll_hal(u8 op_mode, amb_clock_frequency_t set_freq)
{
	int chk_count = 0;
	amb_hal_success_t success;
	amb_clock_source_t clk_src_sel;

	switch (op_mode) {
#if (CHIP_REV == A5S)
		case AUC_CLK_ONCHIP_PLL_SP_CLK:
			clk_src_sel = AMB_PLL_REFERENCE_CLOCK_SOURCE_SP_CLK;
			break;
#endif
		case AUC_CLK_ONCHIP_PLL_CLK_SI:
			clk_src_sel = AMB_PLL_REFERENCE_CLOCK_SOURCE_CLK_SI;
			break;
		case AUC_CLK_ONCHIP_PLL_LVDS_IDSP_SCLK:
			clk_src_sel = AMB_PLL_REFERENCE_CLOCK_SOURCE_LVDS_IDSP_SCLK;
			break;
		case AUC_CLK_EXTERNAL:
			clk_src_sel = AMB_EXTERNAL_CLOCK_SOURCE;
			break;
		default:
		case AUC_CLK_ONCHIP_PLL_27MHZ:
			clk_src_sel = AMB_PLL_REFERENCE_CLOCK_SOURCE_CLK_REF;
			break;
	}

	if (clk_src_sel == AMB_PLL_REFERENCE_CLOCK_SOURCE_CLK_SI ||
		clk_src_sel == AMB_PLL_REFERENCE_CLOCK_SOURCE_LVDS_IDSP_SCLK) {
		success = amb_set_audio_clock_source(HAL_BASE_VP, clk_src_sel, set_freq);
	} else {
		success = amb_set_audio_clock_source(HAL_BASE_VP, clk_src_sel, 0);
	}

	if (success == AMB_HAL_SUCCESS) {
		if(clk_src_sel != AMB_EXTERNAL_CLOCK_SOURCE) {
			success = amb_set_audio_clock_frequency(HAL_BASE_VP, set_freq);
			while (success == AMB_HAL_RETRY) {
				// do something else or sleep
				dly_tsk(1);
				chk_count ++;
				success = amb_set_audio_clock_frequency(HAL_BASE_VP, set_freq);
				if (chk_count == 100) {
					printk("a previous audio pll frequency change request is still outstanding");
					break;
				}
			};
			if (success == AMB_HAL_FAIL)
				printk("new audio pll frequency requested is not supported");
		}
	} else {
		printk("new audio pll reference clock source is not supported");
	}

}

void rct_set_audio_pll_fs(u8 op_mode, u8 mclk_idx)
{
	amb_clock_frequency_t set_freq;
	set_freq = (amb_clock_frequency_t) mclk_table[mclk_idx];
	rct_set_audio_pll_hal(op_mode, set_freq);
}

#elif (RCT_AUDIO_PLL_CONF_MODE == 2)
static const u32 audiopll_config_table[AUC_PLL_CLKRATE_NUM][MAX_MCLK_IDX_NUM][4] =
{
	{	// SP_CLK PLL REFERENCE   =  13.5MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x098ead66,	0x1d031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x1ad9ff8c,	0x1d021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x098ead66,	0x1d021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x1b089a03,	0x1d031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x098ead66,	0x1d031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x1b089a03,	0x1d031108},	//8.4672MHz
		{0x00010019,	0x00010001,	0x573ac902,	0x1d031108},	//8.192MHz
		{0x00010021,	0x00010001,	0x098ead66,	0x1d031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x1b089a03,	0x1d031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x098ead66,	0x1d031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x1b089a03,	0x1d031108},	//4.2336MHz
		{0x00010028,	0x00010001,	0x573ac902,	0x1d041108},	//4.096MHz
		{0x00010026,	0x00010001,	0x43cfc21b,	0x1d061108},	//3.0720MHz
		{0x0001002b,	0x00010001,	0x76eba813,	0x1e061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x098ead66,	0x1d081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  24MHz, VCO = 800~850MHz
		{0x0001000b,	0x00010001,	0xe560418a,	0x10031108},	//18.432MHz
		{0x00010010,	0x00010001,	0xef1a9fbf,	0x10021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0xe560418a,	0x10021108},	//12.288MHz
		{0x00010012,	0x00010001,	0xef34d6a2,	0x10031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0xe560418a,	0x10031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0xef34d6a2,	0x10031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0xe560418a,	0x10021108},	//8.192MHz
		{0x00010021,	0x00010001,	0xe560418a,	0x10031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0xef34d6a2,	0x10031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0xe560418a,	0x10031108},	//4.608MHz
		{0x00010030,	0x00010001,	0xef34d6a2,	0x10031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0xe560418a,	0x10051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0xe560418a,	0x10051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0xe0275255,	0x10061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0xe560418a,	0x10081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  27MHz, VCO = 800MHz~850MHz
		{0x0001000f,	0x00010001,	0x5c28f5c3,	0x0e021108},	//18.432MHz
		{0x00010010,	0x00010001,	0x0d6cffc6,	0x0e021108},	//16.9344MHz
		{0x00010011,	0x00010001,	0x7949801e,	0x0e031108},	//12.288MHz
		{0x00010012,	0x00010001,	0x0d844d02,	0x0e031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x04c756b3,	0x0e031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x0d844d02,	0x0e031108},	//8.4672MHz
		{0x00010019,	0x00010001,	0x2b9d6481,	0x0e031108},	//8.192MHz
		{0x00010021,	0x00010001,	0x04c756b3,	0x0e031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x0d844d02,	0x0e031108},	//5.6448MHz
		{0x00010024,	0x00010001,	0x5c28f5c3,	0x0e041108},	//4.608MHz
		{0x00010027,	0x00010001,	0x49ba5e36,	0x0e041108},	//4.2336MHz
		{0x00010028,	0x00010001,	0x2b9d6481,	0x0e041108},	//4.096MHz
		{0x00010026,	0x00010001,	0x21e7e10e,	0x0e061108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x0022f3da,	0x0e061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x04c756b3,	0x0e081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  27*1.001MHz, VCO = 800MHz~850MHz
		{0x0001000f,	0x00010001,	0x5c28f5c3,	0x0e021108},	//18.432MHz
		{0x00010010,	0x00010001,	0x0d6cffc6,	0x0e021108},	//16.9344MHz
		{0x00010011,	0x00010001,	0x75546c8d,	0x0e031108},	//12.288MHz
		{0x00010012,	0x00010001,	0x09aac939,	0x0e031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x04c756b3,	0x0e031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x0d844d02,	0x0e031108},	//8.4672MHz
		{0x00010019,	0x00010001,	0x27bc2e30,	0x0e031108},	//8.192MHz
		{0x00010021,	0x00010001,	0x00f00f01,	0x0e031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x09aac939,	0x0e031108},	//5.6448MHz
		{0x00010024,	0x00010001,	0x5c28f5c3,	0x0e041108},	//4.608MHz
		{0x00010027,	0x00010001,	0x49ba5e36,	0x0e041108},	//4.2336MHz
		{0x00010028,	0x00010001,	0x27bc2e30,	0x0e041108},	//4.096MHz
		{0x00010026,	0x00010001,	0x1e092664,	0x0e061108},	//3.0720MHz
		{0x0001002a,	0x00010001,	0x59de5815,	0x0e061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x00f00f01,	0x0e081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  48MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x72b020c5,	0x07031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x778d4fe0,	0x07021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x72b020c5,	0x07021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x779a6b51,	0x07031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x72b020c5,	0x07031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x779a6b51,	0x07031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x72b020c5,	0x07021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x72b020c5,	0x07031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x779a6b51,	0x07031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x72b020c5,	0x07031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x779a6b51,	0x07031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x72b020c5,	0x07051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x72b020c5,	0x07051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x7013a92b,	0x07061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x72b020c5,	0x07081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  74.25/1.001MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x777fdb2b,	0x04031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x7aa5a0e8,	0x04021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x777fdb2b,	0x04021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x7aae1c40,	0x04031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x777fdb2b,	0x04031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x7aae1c40,	0x04031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x777fdb2b,	0x04021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x777fdb2b,	0x04031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x7aae1c40,	0x04031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x777fdb2b,	0x04031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x7aae1c40,	0x04031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x777fdb2b,	0x04051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x777fdb2b,	0x04051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x75cf4899,	0x04061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x777fdb2b,	0x04081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  74.25MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x7619f0fc,	0x04031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x793ee8a5,	0x04021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x7619f0fc,	0x04021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x794761d2,	0x04031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x7619f0fc,	0x04031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x794761d2,	0x04031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x7619f0fc,	0x04021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x7619f0fc,	0x04031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x794761d2,	0x04031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x7619f0fc,	0x04031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x794761d2,	0x04031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x7619f0fc,	0x04051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x7619f0fc,	0x04051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x7469cd0a,	0x04061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x7619f0fc,	0x04081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  96MHz/1.001, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x3a6ce347,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x3cdc1a36,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x3a6ce347,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x3ce2a99c,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x3a6ce347,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x3ce2a99c,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x3a6ce347,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x3a6ce347,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x3ce2a99c,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x3a6ce347,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x3ce2a99c,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x3a6ce347,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x3a6ce347,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x391e51e9,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x3a6ce347,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  96MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x39581063,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x3bc6a7f0,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x39581063,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x3bcd35a9,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x39581063,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x3bcd35a9,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x39581063,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x39581063,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x3bcd35a9,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x39581063,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x3bcd35a9,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x39581063,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x39581063,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x3809d496,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x39581063,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  99MHz/1.001, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x199fe425,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x1bfc3872,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x199fe425,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x1c0294f5,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x199fe425,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x1c0294f5,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x199fe425,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x199fe425,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x1c0294f5,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x199fe425,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x1c0294f5,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x199fe425,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x199fe425,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x185b7637,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x199fe425,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  99MHz, VCO = 800MHz~850MHz
		{0x0001000b,	0x00010001,	0x189374bd,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x1aef2e7c,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x189374bd,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x1af5895e,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x189374bd,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x1af5895e,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x189374bd,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x189374bd,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x1af5895e,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x189374bd,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x1af5895e,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x189374bd,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x189374bd,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x174f59c7,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x189374bd,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  148.5/1.001MHz, VCO = 800~850 MHz
		{0x0001000b,	0x00010001,	0xbbbfed96,	0x02031108},	//18.432MHz
		{0x00010010,	0x00010001,	0xbd52d074,	0x02021108},	//16.9344MHz
		{0x00010021,	0x00010001,	0xbbbfed96,	0x02021108},	//12.288MHz
		{0x00010018,	0x00010001,	0xbd570e20,	0x02021108},	//11.2896MHz
		{0x00010016,	0x00010001,	0xbbbfed96,	0x02031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0xbd570e20,	0x02031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0xbbbfed96,	0x02021108},	//8.192MHz
		{0x00010021,	0x00010001,	0xbbbfed96,	0x02031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0xbd570e20,	0x02031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0xbbbfed96,	0x02031108},	//4.608MHz
		{0x00010030,	0x00010001,	0xbd570e20,	0x02031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0xbbbfed96,	0x02051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0xbbbfed96,	0x02051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0xbae7a44d,	0x02061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0xbbbfed96,	0x02081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  148.5MHz, VCO = 800~850 MHz
		{0x0001000b,	0x00010001,	0xbb0cf87e,	0x02031108},	//18.432MHz
		{0x00010010,	0x00010001,	0xbc9f7453,	0x02021108},	//16.9344MHz
		{0x00010021,	0x00010001,	0xbb0cf87e,	0x02021108},	//12.288MHz
		{0x00010018,	0x00010001,	0xbca3b0e9,	0x02021108},	//11.2896MHz
		{0x00010016,	0x00010001,	0xbb0cf87e,	0x02031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0xbca3b0e9,	0x02031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0xbb0cf87e,	0x02021108},	//8.192MHz
		{0x00010021,	0x00010001,	0xbb0cf87e,	0x02031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0xbca3b0e9,	0x02031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0xbb0cf87e,	0x02031108},	//4.608MHz
		{0x00010030,	0x00010001,	0xbca3b0e9,	0x02031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0xbb0cf87e,	0x02051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0xbb0cf87e,	0x02051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0xba34e685,	0x02061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0xbb0cf87e,	0x02081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  90.69/1.001MHz, VCO = 800~850MHz
		{0x0001000b,	0x00010001,	0x79cd736b,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x7c6127c4,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x79cd736b,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x7c68197f,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x79cd736b,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x7c68197f,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x79cd736b,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x79cd736b,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x7c68197f,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x79cd736b,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x7c68197f,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x79cd736b,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x79cd736b,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x786b4b30,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x79cd736b,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  90.69MHz, VCO = 800~850MHz
		{0x0001000b,	0x00010001,	0x78a86b73,	0x03031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x7b3b7716,	0x03021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x78a86b73,	0x03021108},	//12.288MHz
		{0x00010012,	0x00010001,	0x7b42670a,	0x03031108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x78a86b73,	0x03031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x7b42670a,	0x03031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x78a86b73,	0x03021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x78a86b73,	0x03031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x7b42670a,	0x03031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x78a86b73,	0x03031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x7b42670a,	0x03031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x78a86b73,	0x03051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x78a86b73,	0x03051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x77469dcb,	0x03061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x78a86b73,	0x03081108}	//2.048MHz
	},

	{	// SP_CLK PLL REFERENCE   =  65/1.001MHz, VCO = 800~850MHz
		{0x0001000b,	0x00010001,	0x3ea8b3be,	0x05031108},	//18.432MHz
		{0x00010010,	0x00010001,	0x42412464,	0x05021108},	//16.9344MHz
		{0x00010016,	0x00010001,	0x3ea8b3be,	0x05021108},	//12.288MHz
		{0x00010018,	0x00010001,	0x424ad4bc,	0x05021108},	//11.2896MHz
		{0x00010016,	0x00010001,	0x3ea8b3be,	0x05031108},	//9.2160MHz
		{0x00010018,	0x00010001,	0x424ad4bc,	0x05031108},	//8.4672MHz
		{0x00010021,	0x00010001,	0x3ea8b3be,	0x05021108},	//8.192MHz
		{0x00010021,	0x00010001,	0x3ea8b3be,	0x05031108},	//6.1440MHz
		{0x00010024,	0x00010001,	0x424ad4bc,	0x05031108},	//5.6448MHz
		{0x0001002c,	0x00010001,	0x3ea8b3be,	0x05031108},	//4.608MHz
		{0x00010030,	0x00010001,	0x424ad4bc,	0x05031108},	//4.2336MHz
		{0x00010021,	0x00010001,	0x3ea8b3be,	0x05051108},	//4.096MHz
		{0x0001002c,	0x00010001,	0x3ea8b3be,	0x05051108},	//3.0720MHz
		{0x00010029,	0x00010001,	0x3cba9237,	0x05061108},	//2.8224MHz
		{0x0001002c,	0x00010001,	0x3ea8b3be,	0x05081108}	//2.048MHz
	}
};

static u32 get_audio_pll_fs_index(u32 pll_fs)
{
	switch (pll_fs) {
		case AUC_PLL_CLKRATE_1350:	return 0;
		case AUC_PLL_CLKRATE_2400:	return 1;
		case AUC_PLL_CLKRATE_2700:	return 2;
		case AUC_PLL_CLKRATE_2700_M1:	return 3;
		case AUC_PLL_CLKRATE_4800:	return 4;
		case AUC_PLL_CLKRATE_7425_D1:	return 5;
		case AUC_PLL_CLKRATE_7425:	return 6;
		case AUC_PLL_CLKRATE_9600_D1:	return 7;
		case AUC_PLL_CLKRATE_9600:	return 8;
		case AUC_PLL_CLKRATE_9900_D1:	return 9;
		case AUC_PLL_CLKRATE_9900:	return 10;
		case AUC_PLL_CLKRATE_1485_D1:   return 11;
		case AUC_PLL_CLKRATE_1485:      return 12;
		case AUC_PLL_CLKRATE_9069_D1:   return 13;
		case AUC_PLL_CLKRATE_9069:      return 14;
		case AUC_PLL_CLKRATE_6500_D1: return 15;
	}
	printk("Unsupport Audio PLL Source CLK Rate");
//	K_ASSERT(((0x1)==(0x0)));

	return 0;
}

void rct_set_audio_pll_reset(void)
{
	//int i;
	u32 ctrl_reg_val;
	ctrl_reg_val = readl(PLL_AUDIO_CTRL_REG);
	writel(PLL_AUDIO_CTRL_REG, ctrl_reg_val|(0x1<<4));
	//for(i=0;i<100000;i++);
	dly_tsk(1);
	writel(PLL_AUDIO_CTRL_REG, ctrl_reg_val&0xffffffef);
}

static u32 last_ref_clk = REF_CLK_FREQ;

void rct_set_audio_pll_fs_basic(u32 clk_ref,u8 mclk_idx)
{
	//int i;
	u32 ctrl;

	//1.set the audio control register, disable enable bit
	ctrl = readl(PLL_AUDIO_CTRL_REG);
	ctrl &= ~0x01;
	writel(PLL_AUDIO_CTRL_REG, ctrl);
	writel(PLL_AUDIO_CTRL_REG, audiopll_config_table[clk_ref][mclk_idx][3]);
	//2.reset audio pll
	rct_set_audio_pll_reset();
	//3.set the fractional part
	writel(PLL_AUDIO_FRAC_REG, audiopll_config_table[clk_ref][mclk_idx][2]);
	//4.set audio post scaler before pre scaler
	writel(SCALER_AUDIO_REG, (audiopll_config_table[clk_ref][mclk_idx][0]+1));
	dly_tsk(1);
	writel(SCALER_AUDIO_REG, audiopll_config_table[clk_ref][mclk_idx][0]);
	//for(i=0;i<10000;i++);//dly around 0.1ms
	dly_tsk(1);
	//5.set audio pre scaler after post scaler
	writel(SCALER_AUDIO_PRE_REG, (audiopll_config_table[clk_ref][mclk_idx][1]+1));
	dly_tsk(1);
	writel(SCALER_AUDIO_PRE_REG, audiopll_config_table[clk_ref][mclk_idx][1]);
	//6.enable audio PLL
	writel(PLL_AUDIO_CTRL_REG, (audiopll_config_table[clk_ref][mclk_idx][3]+1));
	dly_tsk(1);
}

void rct_set_audio_pll_fs(u8 op_mode, u8 mclk_idx)
{
	int i;
	u32 clk_ref = 0;
	u32 cur_ref_clk;
	u8 switch_flag = 0;
	u32 aud_lock;
	u32 aud_lock_cnt = 0;
	u32 ctrl;

	// RCT set audio PLL ref, CLK source, and MCLK
	if(op_mode < AUC_CLK_EXTERNAL)	{
		if (op_mode == AUC_CLK_ONCHIP_PLL_27MHZ) {
			cur_ref_clk = REF_CLK_FREQ;
			switch_flag = (last_ref_clk > cur_ref_clk)?0:1;
			clk_ref = get_audio_pll_fs_index(REF_CLK_FREQ);
			last_ref_clk = REF_CLK_FREQ;
		} else {
			cur_ref_clk = get_spclk_freq_hz();
			switch_flag = (last_ref_clk > cur_ref_clk)?0:1;
			clk_ref = get_audio_pll_fs_index(cur_ref_clk);
			last_ref_clk = cur_ref_clk;
		}
		if (!switch_flag) {
			writeb(USE_CLK_SI_4_AU_REG, 	audiopll_ref_table[op_mode][1]);
			writeb(CLK_REF_AU_EXTERNAL_REG,	audiopll_ref_table[op_mode][0]);
			writeb(USE_EXTERNAL_CLK_AU_REG,	0x0);
		}

		//1.set the audio control register, disable enable bit
		ctrl = readl(PLL_AUDIO_CTRL_REG);
		ctrl &= ~0x01;
		writel(PLL_AUDIO_CTRL_REG, ctrl);
		writel(PLL_AUDIO_CTRL_REG, audiopll_config_table[clk_ref][mclk_idx][3]);
		//2.reset audio pll
		rct_set_audio_pll_reset();
		//3.set the fractional part
		writel(PLL_AUDIO_FRAC_REG, audiopll_config_table[clk_ref][mclk_idx][2]);
		//4.set audio post scaler before pre scaler
		writel(SCALER_AUDIO_REG, (audiopll_config_table[clk_ref][mclk_idx][0]+1));
		dly_tsk(1);
		writel(SCALER_AUDIO_REG, audiopll_config_table[clk_ref][mclk_idx][0]);
		//for(i=0;i<10000;i++);//dly around 0.1ms
		dly_tsk(1);
		//5.set audio pre scaler after post scaler
		writel(SCALER_AUDIO_PRE_REG, (audiopll_config_table[clk_ref][mclk_idx][1]+1));
		dly_tsk(1);
		writel(SCALER_AUDIO_PRE_REG, audiopll_config_table[clk_ref][mclk_idx][1]);
		//6.enable audio PLL
		writel(PLL_AUDIO_CTRL_REG, (audiopll_config_table[clk_ref][mclk_idx][3]+1));
		dly_tsk(1);

		if (switch_flag) {
			writeb(USE_CLK_SI_4_AU_REG, 	audiopll_ref_table[op_mode][1]);
			writeb(CLK_REF_AU_EXTERNAL_REG,	audiopll_ref_table[op_mode][0]);
			writeb(USE_EXTERNAL_CLK_AU_REG,	0x0);
		}


		for (i = 0; i < 2000; i++) {
			aud_lock = readl(PLL_LOCK_REG);
			aud_lock >>= 7;
			aud_lock &= 0x00000001;
			if(aud_lock == 0)
				rct_set_audio_pll_fs_basic(clk_ref,mclk_idx);
			else
				break;

			aud_lock_cnt ++;
		}

		//printk("reprogram audio pll lock %d times",aud_lock_cnt);
		if(i == 2000 && aud_lock == 0){
			printk("audio pll lock fail!!!!!!!!!!!");
		}

	} else { //External Audio Clock Source
		writeb(USE_EXTERNAL_CLK_AU_REG,	0x1);
	}

	rct_set_audio_freq_hz(mclk_table[mclk_idx]);
}

#elif (RCT_AUDIO_PLL_CONF_MODE == 1)
static const u32 audiopll_config_table[AUC_PLL_CLKRATE_NUM][MAX_MCLK_IDX_NUM][4] =
{
		// SP_CLK PLL REFERENCE   =  13.5MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 13.5MHz
		{0x0000000A, 0x00000001, 0x1B01B102, 0x00004E82},//18.432MHz
		{0x0000000A, 0x00000001, 0x1901B102, 0x00001687},//16.9344MHz
		{0x0000000E, 0x00000001, 0x3201F102, 0x0000F8F2},//12.288MHz
		{0x0000000E, 0x00000001, 0x2E01F102, 0x0000D4B8},//11.2896MHz
		{0x00000014, 0x00000001, 0x1B01B102, 0x00004E82},//9.2160MHz
		{0x00000014, 0x00000001, 0x1901B102, 0x00001687},//8.4672MHz
		{0x00000015, 0x00000001, 0x3201F102, 0x0000F8F2},//8.192MHz
		{0x0000001C, 0x00000001, 0x3201F102, 0x0000F8F2},//6.1440MHz
	  	{0x0000001C, 0x00000001, 0x2E01F102, 0x0000D4B8},//5.6448MHz
		{0x00000028, 0x00000001, 0x1B01B102, 0x00004E82},//4.608MHz
		{0x00000028, 0x00000001, 0x1901B102, 0x00001687},//4.2336MHz
		{0x0000002A, 0x00000001, 0x3201F102, 0x0000F8F2},//4.096MHz
		{0x00000038, 0x00000001, 0x3201F102, 0x0000F8F2},//3.0720MHz
		{0x00000038, 0x00000001, 0x2E01F102, 0x0000D4B8},//2.8224MHz
		{0x00000054, 0x00000001, 0x3201F102, 0x0000F8F2} //2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  24MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 24MHz
		{0x0000000A, 0x00000001, 0x1E01F102, 0x0000B852},//18.432MHz
		{0x0000000A, 0x00000001, 0x1C01F102, 0x00003958},//16.9344MHz
		{0x0000000E, 0x00000001, 0x1C01F102, 0x0000AC08},//12.288MHz
		{0x0000000E, 0x00000001, 0x1A01F102, 0x000057A8},//11.2896MHz
		{0x00000014, 0x00000001, 0x1E01F102, 0x0000B852},//9.2160MHz
		{0x00000014, 0x00000001, 0x1C01F102, 0x00003958},//8.4672MHz
		{0x00000015, 0x00000001, 0x1C01F102, 0x0000AC08},//8.192MHz
		{0x0000001C, 0x00000001, 0x1C01F102, 0x0000AC08},//6.1440MHz
		{0x0000001C, 0x00000001, 0x1A01F102, 0x000057A8},//5.6448MHz
		{0x00000028, 0x00000001, 0x1E01F102, 0x0000B852},//4.608MHz
		{0x00000028, 0x00000001, 0x1C01F102, 0x00003958},//4.2336MHz
		{0x0000002A, 0x00000001, 0x1C01F102, 0x0000AC08},//4.096MHz
		{0x00000038, 0x00000001, 0x1C01F102, 0x0000AC08},//3.0720MHz
		{0x00000038, 0x00000001, 0x1A01F102, 0x000057A8},//2.8224MHz
		{0x00000054, 0x00000001, 0x1C01F102, 0x0000AC08} //2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  27MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 27M
		{0x0000000A, 0x00000001, 0x1B01F102, 0x00004E82},//18.432MHz
		{0x0000000A, 0x00000001, 0x1901F102, 0x00001687},//16.9344MHz
		{0x0000000E, 0x00000001, 0x1901F102, 0x00007C79},//12.288MHz
		{0x0000000E, 0x00000001, 0x1701F102, 0x00006A5C},//11.2896MHz
		{0x00000014, 0x00000001, 0x1B01F102, 0x00004E82},//9.2160MHz
		{0x00000014, 0x00000001, 0x1901F102, 0x00001687},//8.4672MHz
		{0x00000015, 0x00000001, 0x1901F102, 0x00007C79},//8.192MHz
		{0x0000001C, 0x00000001, 0x1901F102, 0x00007C79},//6.1440MHz
		{0x0000001C, 0x00000001, 0x1701F102, 0x00006A5C},//5.6448MHz
		{0x00000028, 0x00000001, 0x1B01F102, 0x00004E82},//4.608MHz
		{0x00000028, 0x00000001, 0x1901F102, 0x00001687},//4.2336MHz
		{0x0000002A, 0x00000001, 0x1901F102, 0x00007C79},//4.096MHz
		{0x00000038, 0x00000001, 0x1901F102, 0x00007C79},//3.0720MHz
		{0x00000038, 0x00000001, 0x1701F102, 0x00006A5C},//2.8224MHz
		{0x00000054, 0x00000001, 0x1901F102, 0x00007C79} //2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  27*1.001MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 27*1.001MHz
		{0x0000000A, 0x00000001, 0x1B01F102, 0x00004786},//18.432MHz
		{0x0000000A, 0x00000001, 0x1901F102, 0x0000101D},//16.9344MHz
		{0x0000000E, 0x00000001, 0x1901F102, 0x000075F4},//12.288MHz
		{0x0000000E, 0x00000001, 0x1701F102, 0x0000645F},//11.2896MHz
		{0x00000014, 0x00000001, 0x1B01F102, 0x00004786},//9.2160MHz
		{0x00000014, 0x00000001, 0x1901F102, 0x0000101D},//8.4672MHz
		{0x00000015, 0x00000001, 0x1901F102, 0x000075F4},//8.192MHz
		{0x0000001C, 0x00000001, 0x1901F102, 0x000075F4},//6.1440MHz
		{0x0000001C, 0x00000001, 0x1701F102, 0x0000645F},//5.6448MHz
		{0x00000028, 0x00000001, 0x1B01F102, 0x00004786},//4.608MHz
		{0x00000028, 0x00000001, 0x1901F102, 0x0000101D},//4.2336MHz
		{0x0000002A, 0x00000001, 0x1901F102, 0x000075F4},//4.096MHz
		{0x00000038, 0x00000001, 0x1901F102, 0x000075F4},//3.0720MHz
		{0x00000038, 0x00000001, 0x1701F102, 0x0000645F},//2.8224MHz
		{0x00000054, 0x00000001, 0x1901F102, 0x000075F4}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  48MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 48M
		{0x0000000A, 0x00000001, 0x0701B102, 0x0000AE14},//18.432MHz
		{0x0000000A, 0x00000001, 0x0701B102, 0x00000E56},//16.9344MHz
		{0x0000000E, 0x00000001, 0x0701B102, 0x00002B02},//12.288MHz
	  	{0x0000000E, 0x00000001, 0x0601B102, 0x000095EA},//11.2896MHz
		{0x00000014, 0x00000001, 0x0701B102, 0x0000AE14},//9.2160MHz
		{0x00000014, 0x00000001, 0x0701B102, 0x00000E56},//8.4672MHz
		{0x00000015, 0x00000001, 0x0701B102, 0x00002B02},//8.192MHz
		{0x0000001C, 0x00000001, 0x0701B102, 0x00002B02},//6.1440MHz
		{0x0000001C, 0x00000001, 0x0601B102, 0x000095EA},//5.6448MHz
		{0x00000028, 0x00000001, 0x0701B102, 0x0000AE14},//4.608MHz
		{0x00000028, 0x00000001, 0x0701B102, 0x00000E56},//4.2336MHz
		{0x0000002A, 0x00000001, 0x0701B102, 0x00002B02},//4.096MHz
		{0x00000038, 0x00000001, 0x0701B102, 0x00002B02},//3.0720MHz
		{0x00000038, 0x00000001, 0x0601B102, 0x000095EA},//2.8224MHz
		{0x00000054, 0x00000001, 0x0701B102, 0x00002B02}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  74.25/1.001MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 74.25/1.001/2MHz
		{0x0000000A, 0x00000002, 0x1301F102, 0x0000E117},//18.432MHz
		{0x0000000A, 0x00000002, 0x1201F102, 0x0000439A},//16.9344MHz
		{0x0000000E, 0x00000002, 0x1201F102, 0x00008DD1},//12.288MHz,
		{0x0000000E, 0x00000002, 0x1101F102, 0x00000BE5},//11.2896MHz,
		{0x00000014, 0x00000002, 0x1301F102, 0x0000E117},//9.2160MHz
		{0x00000014, 0x00000002, 0x1201F102, 0x0000439A},//8.4672MHz
		{0x00000015, 0x00000002, 0x1201F102, 0x00008DD1},//8.192MHz,
		{0x0000001C, 0x00000002, 0x1201F102, 0x00008DD1},//6.1440MHz,
		{0x0000001C, 0x00000002, 0x1101F102, 0x00000BE5},//5.6448MHz,
		{0x00000028, 0x00000002, 0x1301F102, 0x0000E117},//4.608MHz
		{0x00000028, 0x00000002, 0x1201F102, 0x0000439A},//4.2336MHz
	  	{0x0000002A, 0x00000002, 0x1201F102, 0x00008DD1},//4.096MHz,
	  	{0x00000038, 0x00000002, 0x1201F102, 0x00008DD1},//3.0720MHz,
	  	{0x00000038, 0x00000002, 0x1101F102, 0x00000BE5},//2.8224MHz,
	  	{0x00000054, 0x00000002, 0x1201F102, 0x00008DD1}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  74.25MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 74.25/2MHz
		{0x0000000A, 0x00000002, 0x1301F102, 0x0000DC01},//18.432MHz
		{0x0000000A, 0x00000002, 0x1201F102, 0x00003EEE},//16.9344MHz
		{0x0000000E, 0x00000002, 0x1201F102, 0x00008912},//12.288MHz,
		{0x0000000E, 0x00000002, 0x1101F102, 0x00000789},//11.2896MHz,
		{0x00000014, 0x00000002, 0x1301F102, 0x0000DC01},//9.2160MHz
		{0x00000014, 0x00000002, 0x1201F102, 0x00003EEE},//8.4672MHz
		{0x00000015, 0x00000002, 0x1201F102, 0x00008912},//8.192MHz,
		{0x0000001C, 0x00000002, 0x1201F102, 0x00008912},//6.1440MHz,
		{0x0000001C, 0x00000002, 0x1101F102, 0x00000789},//5.6448MHz,
		{0x00000028, 0x00000002, 0x1301F102, 0x0000DC01},//4.608MHz
		{0x00000028, 0x00000002, 0x1201F102, 0x00003EEE},//4.2336MHz
		{0x0000002A, 0x00000002, 0x1201F102, 0x00008912},//4.096MHz,
		{0x00000038, 0x00000002, 0x1201F102, 0x00008912},//3.0720MHz,
		{0x00000038, 0x00000002, 0x1101F102, 0x00000789},//2.8224MHz,
		{0x00000054, 0x00000002, 0x1201F102, 0x00008912}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  96MHz/1.001 = 95904096
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 96MHz/1.001/2
		{0x0000000A, 0x00000002, 0x0701B102, 0x0000B00C},//18.432MHz
		{0x0000000A, 0x00000002, 0x0701B102, 0x00001024},//16.9344MHz
		{0x0000000E, 0x00000002, 0x0701B102, 0x00002CD8},//12.288MHz,
		{0x0000000E, 0x00000002, 0x0601B102, 0x00009799},//11.2896MHz,
		{0x00000014, 0x00000002, 0x0701B102, 0x0000B00C},//9.2160MHz
		{0x00000014, 0x00000002, 0x0701B102, 0x00001024},//8.4672MHz
		{0x00000015, 0x00000002, 0x0701B102, 0x00002CD8},//8.192MHz,
		{0x0000001C, 0x00000002, 0x0701B102, 0x00002CD8},//6.1440MHz,
		{0x0000001C, 0x00000002, 0x0601B102, 0x00009799},//5.6448MHz,
		{0x00000028, 0x00000002, 0x0701B102, 0x0000B00C},//4.608MHz
		{0x00000028, 0x00000002, 0x0701B102, 0x00001024},//4.2336MHz
		{0x0000002A, 0x00000002, 0x0701B102, 0x00002CD8},//4.096MHz,
		{0x00000038, 0x00000002, 0x0701B102, 0x00002CD8},//3.0720MHz,
		{0x00000038, 0x00000002, 0x0601B102, 0x00009799},//2.8224MHz,
		{0x00000054, 0x00000002, 0x0701B102, 0x00002CD8}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  96MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 96MHz/2
		{0x0000000A, 0x00000002, 0x0701B102, 0x0000AE14},//18.432MHz
		{0x0000000A, 0x00000002, 0x0701B102, 0x00000E56},//16.9344MHz
		{0x0000000E, 0x00000002, 0x0701B102, 0x00002B02},//12.288MHz,
		{0x0000000E, 0x00000002, 0x0601B102, 0x000095EA},//11.2896MHz,
		{0x00000014, 0x00000002, 0x0701B102, 0x0000AE14},//9.2160MHz
		{0x00000014, 0x00000002, 0x0701B102, 0x00000E56},//8.4672MHz
		{0x00000015, 0x00000002, 0x0701B102, 0x00002B02},//8.192MHz,
		{0x0000001C, 0x00000002, 0x0701B102, 0x00002B02},//6.1440MHz,
		{0x0000001C, 0x00000002, 0x0601B102, 0x000095EA},//5.6448MHz,
		{0x00000028, 0x00000002, 0x0701B102, 0x0000AE14},//4.608MHz
		{0x00000028, 0x00000002, 0x0701B102, 0x00000E56},//4.2336MHz
		{0x0000002A, 0x00000002, 0x0701B102, 0x00002B02},//4.096MHz,
		{0x00000038, 0x00000002, 0x0701B102, 0x00002B02},//3.0720MHz,
		{0x00000038, 0x00000002, 0x0601B102, 0x000095EA},//2.8224MHz,
		{0x00000054, 0x00000002, 0x0701B102, 0x00002B02}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  99MHz/1.001
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 99MHz/1.001
		{0x0000000A, 0x00000003, 0x1601F102, 0x00005D3A},//18.432MHz
		{0x0000000A, 0x00000003, 0x1401F102, 0x00008C0D},//16.9344MHz
		{0x0000000E, 0x00000003, 0x1401F102, 0x0000DF8B},//12.288MHz,
		{0x0000000E, 0x00000003, 0x1301F102, 0x00002D61},//11.2896MHz
		{0x00000014, 0x00000003, 0x1601F102, 0x00005D3A},//9.2160MHz
		{0x00000014, 0x00000003, 0x1401F102, 0x00008C0D},//8.4672MHz
		{0x00000015, 0x00000003, 0x1401F102, 0x0000DF8B},//8.192MHz,
		{0x0000001C, 0x00000003, 0x1401F102, 0x0000DF8B},//6.1440MHz,
		{0x0000001C, 0x00000003, 0x1301F102, 0x00002D61},//5.6448MHz,
		{0x00000028, 0x00000003, 0x1601F102, 0x00005D3A},//4.608MHz
		{0x00000028, 0x00000003, 0x1401F102, 0x00008C0D},//4.2336MHz
		{0x0000002A, 0x00000003, 0x1401F102, 0x0000DF8B},//4.096MHz,
		{0x00000038, 0x00000003, 0x1401F102, 0x0000DF8B},//3.0720MHz,
		{0x00000038, 0x00000003, 0x1301F102, 0x00002D61},//2.8224MHz,
		{0x00000054, 0x00000003, 0x1401F102, 0x0000DF8B}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  99MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 99MHz
		{0x0000000A, 0x00000003, 0x1601F102, 0x00005781},//18.432MHz
		{0x0000000A, 0x00000003, 0x1401F102, 0x000086CC},//16.9344MHz
		{0x0000000E, 0x00000003, 0x1401F102, 0x0000DA35},//12.288MHz,
		{0x0000000E, 0x00000003, 0x1301F102, 0x0000287A},//11.2896MHz
		{0x00000014, 0x00000003, 0x1601F102, 0x00005781},//9.2160MHz
		{0x00000014, 0x00000003, 0x1401F102, 0x000086CC},//8.4672MHz
		{0x00000015, 0x00000003, 0x1401F102, 0x0000DA35},//8.192MHz,
		{0x0000001C, 0x00000003, 0x1401F102, 0x0000DA35},//6.1440MHz,
		{0x0000001C, 0x00000003, 0x1301F102, 0x0000287A},//5.6448MHz,
		{0x00000028, 0x00000003, 0x1601F102, 0x00005781},//4.608MHz
		{0x00000028, 0x00000003, 0x1401F102, 0x000086CC},//4.2336MHz
		{0x0000002A, 0x00000003, 0x1401F102, 0x0000DA35},//4.096MHz,
		{0x00000038, 0x00000003, 0x1401F102, 0x0000DA35},//3.0720MHz,
		{0x00000038, 0x00000003, 0x1301F102, 0x0000287A},//2.8224MHz,
		{0x00000054, 0x00000003, 0x1401F102, 0x0000DA35}//2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  148.5/1.001MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 148.5MHz
	  	{0x0000000A, 0x00000002, 0x0901F102, 0x0000F08B},//18.432MHz
	  	{0x0000000A, 0x00000002, 0x0901F102, 0x000021CD},//16.9344MHz
	  	{0x0000000E, 0x00000002, 0x0901F102, 0x000046E8},//12.288MHz
	  	{0x0000000E, 0x00000002, 0x0801F102, 0x000085F2},//11.2896MHz
	  	{0x00000014, 0x00000002, 0x0901F102, 0x0000F08B},//9.2160MHz
	  	{0x00000014, 0x00000002, 0x0901F102, 0x000021CD},//8.4672MHz
	  	{0x00000015, 0x00000002, 0x0901F102, 0x000046E8},//8.192MHz
	  	{0x0000001C, 0x00000002, 0x0901F102, 0x000046E8},//6.1440MHz
	  	{0x0000001C, 0x00000002, 0x0801F102, 0x000085F2},//5.6448MHz
	  	{0x00000028, 0x00000002, 0x0901F102, 0x0000F08B},//4.608MHz
	  	{0x00000028, 0x00000002, 0x0901F102, 0x000021CD},//4.2336MHz
	  	{0x0000002A, 0x00000002, 0x0901F102, 0x000046E8},//4.096MHz
	  	{0x00000038, 0x00000002, 0x0901F102, 0x000046E8},//3.0720MHz
	  	{0x00000038, 0x00000002, 0x0801F102, 0x000085F2},//2.8224MHz
	  	{0x00000054, 0x00000002, 0x0901F102, 0x000046E8} //2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  148.5MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 148.5MHz
	  	{0x0000000A, 0x00000002, 0x0901F102, 0x0000EE01},//18.432MHz
	  	{0x0000000A, 0x00000002, 0x0901F102, 0x00001F77},//16.9344MHz
	  	{0x0000000E, 0x00000002, 0x0901F102, 0x00004489},//12.288MHz
	  	{0x0000000E, 0x00000002, 0x0801F102, 0x000083C4},//11.2896MHz
	  	{0x00000014, 0x00000002, 0x0901F102, 0x0000EE01},//9.2160MHz
	  	{0x00000014, 0x00000002, 0x0901F102, 0x00001F77},//8.4672MHz
	  	{0x00000015, 0x00000002, 0x0901F102, 0x00004489},//8.192MHz
	  	{0x0000001C, 0x00000002, 0x0901F102, 0x00004489},//6.1440MHz
	  	{0x0000001C, 0x00000002, 0x0801F102, 0x000083C4},//5.6448MHz
	  	{0x00000028, 0x00000002, 0x0901F102, 0x0000EE01},//4.608MHz
	  	{0x00000028, 0x00000002, 0x0901F102, 0x00001F77},//4.2336MHz
	  	{0x0000002A, 0x00000002, 0x0901F102, 0x00004489},//4.096MHz
	  	{0x00000038, 0x00000002, 0x0901F102, 0x00004489},//3.0720MHz
	  	{0x00000038, 0x00000002, 0x0801F102, 0x000083C4},//2.8224MHz
	  	{0x00000054, 0x00000002, 0x0901F102, 0x00004489} //2.048MHz
	},
		/* For alta5262: not verified yet */
		// SP_CLK PLL REFERENCE   =  90.69/1.001MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 90.69/1.001MHz
	  	{0x0000000A, 0x00000002, 0x1001F102, 0x000046E8},//18.432MHz
	  	{0x0000000A, 0x00000002, 0x0E01F102, 0x0000F406},//16.9344MHz
	  	{0x0000000E, 0x00000002, 0x0F01F102, 0x000030C9},//12.288MHz
	  	{0x0000000E, 0x00000002, 0x0D01F102, 0x0000F4D2},//11.2896MHz
	  	{0x00000014, 0x00000002, 0x1001F102, 0x000046E8},//9.2160MHz
	  	{0x00000014, 0x00000002, 0x0E01F102, 0x0000F406},//8.4672MHz
	  	{0x00000015, 0x00000002, 0x0F01F102, 0x000030C9},//8.192MHz
	  	{0x0000001C, 0x00000002, 0x0F01F102, 0x000030C9},//6.1440MHz
	  	{0x0000001C, 0x00000002, 0x0D01F102, 0x0000F4D2},//5.6448MHz
	  	{0x00000028, 0x00000002, 0x1001F102, 0x000046E8},//4.608MHz
	  	{0x00000028, 0x00000002, 0x0E01F102, 0x0000F406},//4.2336MHz
	  	{0x0000002A, 0x00000002, 0x0F01F102, 0x000030C9},//4.096MHz
	  	{0x00000038, 0x00000002, 0x0F01F102, 0x000030C9},//3.0720MHz
	  	{0x00000038, 0x00000002, 0x0D01F102, 0x0000F4D2},//2.8224MHz
	  	{0x00000054, 0x00000002, 0x0F01F102, 0x000030C9} //2.048MHz
	},
		// SP_CLK PLL REFERENCE   =  90.69MHz
	{
		//VCO = 688.1280, 632.2176M, REFCLK_AU = 90.69MHz
	  	{0x0000000A, 0x00000002, 0x1001F102, 0x00004264},//18.432MHz
	  	{0x0000000A, 0x00000002, 0x0E01F102, 0x0000F033},//16.9344MHz
	  	{0x0000000E, 0x00000002, 0x0F01F102, 0x00002CE6},//12.288MHz
	  	{0x0000000E, 0x00000002, 0x0D01F102, 0x0000F140},//11.2896MHz
	  	{0x00000014, 0x00000002, 0x1001F102, 0x00004264},//9.2160MHz
	  	{0x00000014, 0x00000002, 0x0E01F102, 0x0000F033},//8.4672MHz
	  	{0x00000015, 0x00000002, 0x0F01F102, 0x00002CE6},//8.192MHz
	  	{0x0000001C, 0x00000002, 0x0F01F102, 0x00002CE6},//6.1440MHz
	  	{0x0000001C, 0x00000002, 0x0D01F102, 0x0000F140},//5.6448MHz
	  	{0x00000028, 0x00000002, 0x1001F102, 0x00004264},//4.608MHz
	  	{0x00000028, 0x00000002, 0x0E01F102, 0x0000F033},//4.2336MHz
	  	{0x0000002A, 0x00000002, 0x0F01F102, 0x00002CE6},//4.096MHz
	  	{0x00000038, 0x00000002, 0x0F01F102, 0x00002CE6},//3.0720MHz
	  	{0x00000038, 0x00000002, 0x0D01F102, 0x0000F140},//2.8224MHz
	  	{0x00000054, 0x00000002, 0x0F01F102, 0x00002CE6} //2.048MHz
	},

		// SP_CLK PLL REFERENCE = 65/1.001MHz
	{
		//VCO = 774.144~632.2176M, REFCLK_AU = 65/1.001MHz
	  	{0x0000000A, 0x00000001, 0x0B01F102, 0x00005AA7},//18.432MHz
	  	{0x0000000A, 0x00000001, 0x0A01F102, 0x00006E7D},//16.9344MHz
	  	{0x0000000E, 0x00000001, 0x0A01F102, 0x000098E0},//12.288MHz
	  	{0x0000000E, 0x00000001, 0x0901F102, 0x0000BC74},//11.2896MHz
	  	{0x00000014, 0x00000001, 0x0B01F102, 0x00005AA7},//9.2160MHz
	  	{0x00000014, 0x00000001, 0x0A01F102, 0x00006E7D},//8.4672MHz
	  	{0x00000015, 0x00000001, 0x0A01F102, 0x000098E0},//8.192MHz
	  	{0x0000001C, 0x00000001, 0x0A01F102, 0x000098E0},//6.1440MHz
	  	{0x0000001C, 0x00000001, 0x0901F102, 0x0000BC74},//5.6448MHz
	  	{0x0000002A, 0x00000001, 0x0B01F102, 0x0000EBFC},//4.608MHz
	  	{0x0000002A, 0x00000001, 0x0A01F102, 0x0000F403},//4.2336MHz
	  	{0x0000002A, 0x00000001, 0x0A01F102, 0x000098E0},//4.096MHz
	  	{0x00000038, 0x00000001, 0x0A01F102, 0x000098E0},//3.0720MHz
	  	{0x00000038, 0x00000001, 0x0901F102, 0x0000BC74},//2.8224MHz
	  	{0x00000054, 0x00000001, 0x0A01F102, 0x000098E0} //2.048MHz
	}
};

static u32 get_audio_pll_fs_index(u32 pll_fs)
{
	switch (pll_fs) {
		case AUC_PLL_CLKRATE_1350:	return 0;
		case AUC_PLL_CLKRATE_2400:	return 1;
		case AUC_PLL_CLKRATE_2700:	return 2;
		case AUC_PLL_CLKRATE_2700_M1:	return 3;
		case AUC_PLL_CLKRATE_4800:	return 4;
		case AUC_PLL_CLKRATE_7425_D1:	return 5;
		case AUC_PLL_CLKRATE_7425:	return 6;
		case AUC_PLL_CLKRATE_9600_D1:	return 7;
		case AUC_PLL_CLKRATE_9600:	return 8;
		case AUC_PLL_CLKRATE_9900_D1:	return 9;
		case AUC_PLL_CLKRATE_9900:	return 10;
		case AUC_PLL_CLKRATE_1485_D1:   return 11;
		case AUC_PLL_CLKRATE_1485:      return 12;
		case AUC_PLL_CLKRATE_9069_D1:   return 13;
		case AUC_PLL_CLKRATE_9069:      return 14;
		case AUC_PLL_CLKRATE_6500_D1: return 15;
	}
	printk("Unsupport Audio PLL Source CLK Rate");
//	K_ASSERT(((0x1)==(0x0)));

	return 0;
}

void rct_set_audio_pll_reset(void)
{
	u32 ctrl_reg_val;
	ctrl_reg_val = readl(PLL_AUDIO_CTRL_REG);
	writel(PLL_AUDIO_CTRL_REG,	ctrl_reg_val|(0x1<<4));
	rct_alan_zhu_magic_loop(PLL_LOCK_AUDIO);
	writel(PLL_AUDIO_CTRL_REG,	ctrl_reg_val&0xffffffef);
	rct_alan_zhu_magic_loop(PLL_LOCK_AUDIO);
}

void rct_set_audio_pll_fs(u8 op_mode, u8 mclk_idx)
{
	u32 clk_ref;

	// RCT set audio PLL ref, CLK source, and MCLK
	if(op_mode < AUC_CLK_EXTERNAL)	{
		writeb(USE_CLK_SI_4_AU_REG, 	audiopll_ref_table[op_mode][1]);
		writeb(CLK_REF_AU_EXTERNAL_REG,	audiopll_ref_table[op_mode][0]);
		writeb(USE_EXTERNAL_CLK_AU_REG,	0x0);
		if (op_mode == AUC_CLK_ONCHIP_PLL_27MHZ) {
			clk_ref = get_audio_pll_fs_index(REF_CLK_FREQ);
		} else {
			clk_ref = get_audio_pll_fs_index(get_spclk_freq_hz());
		}
		writel(SCALER_AUDIO_PRE_REG, audiopll_config_table[clk_ref][mclk_idx][1]);
		writel(PLL_AUDIO_CTRL_REG, audiopll_config_table[clk_ref][mclk_idx][2]);
		writel(PLL_AUDIO_FRAC_REG, audiopll_config_table[clk_ref][mclk_idx][3]);
		writel(SCALER_AUDIO_REG, audiopll_config_table[clk_ref][mclk_idx][0]);
	} else { //External Audio Clock Source
		writeb(USE_EXTERNAL_CLK_AU_REG,	0x1);
	}
	rct_set_audio_freq_hz(mclk_table[mclk_idx]);
	rct_alan_zhu_magic_loop(PLL_LOCK_AUDIO);

}

#elif (RCT_AUDIO_PLL_CONF_MODE == 0)
static u16 audiopll_config_table[7][3] =
{
	{0x003a, 0x438, 0x24}, {0x4038, 0x43a, 0x24}, {0x403a, 0x438, 0x24},
  	{0x4138, 0x43a, 0x24}, {0x4338, 0x43a, 0x24}, {0x30, 0xe6a, 0x14},
  	//audiopll_config_table[6][] is only for IS2 Test =>27MHz Out
  	{0x003a, 0x438, 0x08}
};

void rct_set_audio_pll_fs(u8 op_mode)
{
	writel(PLL_AU1_REG,
		audiopll_config_table[op_mode][0]);
	writel(PLL_AU2_REG,
		audiopll_config_table[op_mode][1]);
	writel(CG_AU_REG,
		audiopll_config_table[op_mode][2]);

#if	(RCT_SUPPORT_CG_SETTING_CHANGED == 1)
	loc_cpu();
	writel(CG_SETTING_CHANGED_REG, 0x00000000);
	writel(CG_SETTING_CHANGED_REG, 0x00001808);
	writel(CG_SETTING_CHANGED_REG, 0x00000000);
	unl_cpu();
#endif
}
#endif

static struct srcu_notifier_head audio_notifier_list;
static struct notifier_block audio_notify;
static struct ambarella_i2s_interface audio_i2s_intf;

struct ambarella_i2s_interface get_audio_i2s_interface(void)
{
	return audio_i2s_intf;
}
EXPORT_SYMBOL(get_audio_i2s_interface);

static int audio_notify_transition(struct notifier_block *nb,
		unsigned long val, void *data)
{
	switch(val) {
	case AUDIO_NOTIFY_INIT:
		audio_i2s_intf.state = AUDIO_NOTIFY_INIT;
		memcpy(&audio_i2s_intf, data, sizeof(struct ambarella_i2s_interface));
		break;

	case AUDIO_NOTIFY_SETHWPARAMS:
		audio_i2s_intf.state = AUDIO_NOTIFY_SETHWPARAMS;
		memcpy(&audio_i2s_intf, data, sizeof(struct ambarella_i2s_interface));
		break;

	case AUDIO_NOTIFY_REMOVE:
		memset(&audio_i2s_intf, 0, sizeof(struct ambarella_i2s_interface));
		audio_i2s_intf.state = AUDIO_NOTIFY_REMOVE;
		break;
	default:
		audio_i2s_intf.state = AUDIO_NOTIFY_UNKNOWN;
		break;
	}

	return 0;
}

void ambarella_audio_notify_transition (
	struct ambarella_i2s_interface *data, unsigned int type)
{
	srcu_notifier_call_chain(&audio_notifier_list, type, data);
}

int ambarella_audio_register_notifier(struct notifier_block *nb)
{
	return srcu_notifier_chain_register( &audio_notifier_list, nb);
}

int ambarella_audio_unregister_notifier(struct notifier_block *nb)
{
	return srcu_notifier_chain_unregister(&audio_notifier_list, nb);
}

int __init ambarella_init_audio(void)
{
	int retval = 0;

	srcu_init_notifier_head(&audio_notifier_list);

	memset(&audio_i2s_intf, 0, sizeof(struct ambarella_i2s_interface));
	audio_i2s_intf.state = AUDIO_NOTIFY_UNKNOWN;

	audio_notify.notifier_call = audio_notify_transition;
	retval = ambarella_audio_register_notifier(&audio_notify);

	return retval;
}

