/* pidfile.c
 *
 * Functions to assist in the writing and removing of pidfiles.
 *
 * Russ Dill <Russ.Dill@asu.edu> September 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "pidfile.h"
#include "common.h"

static char *saved_pidfile;

static void pidfile_delete(void)
{
  if (saved_pidfile) unlink(saved_pidfile);
}


int pidfile_acquire(const char *pidfile)
{
  int pid_fd;
  if (!pidfile) return -1;

  pid_fd = open(pidfile, O_CREAT | O_WRONLY, 0644);
  if (pid_fd < 0) {
    LOG(LOG_ERR, "Unable to open pidfile %s: %m\n", pidfile);
  } else {
    lockf(pid_fd, F_LOCK, 0);
    if (!saved_pidfile)
      atexit(pidfile_delete);
    saved_pidfile = (char *) pidfile;
  }

  return pid_fd;
}


void pidfile_write_release(int pid_fd)
{
  FILE *out;

  if (pid_fd < 0) return;

  if ((out = fdopen(pid_fd, "w")) != NULL) {
    fprintf(out, "%d\n", getpid());
    fclose(out);
  }
  lockf(pid_fd, F_UNLCK, 0);
  close(pid_fd);
}

/* read_pid
 *
 * Reads the specified pidfile and returns the read pid.
 * 0 is returned if either there's no pidfile, it's empty
 * or no pid can be read.
 */
int read_pid (char *pidfile)
{
  FILE *f;
  int pid;

  if (!(f=fopen(pidfile,"r")))
    return 0;
  fscanf(f,"%d", &pid);
  fclose(f);
  return pid;
}

/* check_pid
 *
 * Reads the pid using read_pid and looks up the pid in the process
 * table (using /proc) to determine if the process already exists. If
 * so 1 is returned, otherwise 0.
 */
int check_pid (char *pidfile)
{
  int pid = read_pid(pidfile);

  /* Amazing ! _I_ am already holding the pid file... */
  if ((!pid) || (pid == getpid ()))
    return 0;

  /*
   * The 'standard' method of doing this is to try and do a 'fake' kill
   * of the process.  If an ESRCH error is returned the process cannot
   * be found -- GW
   */
  /* But... errno is usually changed only on error.. */
  //if (kill(pid, 0) && errno == ESRCH)
  //  return(0);

  return pid;
}

/* write_pid
 *
 * Writes the pid to the specified file. If that fails 0 is
 * returned, otherwise the pid.
 */
int write_pid (char *pidfile)
{
  FILE *f;
  int fd;
  int pid;

  if ( ((fd = open(pidfile, O_RDWR|O_CREAT, 0644)) == -1)
       || ((f = fdopen(fd, "r+")) == NULL) ) {
      fprintf(stderr, "Can't open or create %s.\n", pidfile);
      return 0;
  }

 /* It seems to be acceptable that we do not lock the pid file
  * if we run under Solaris. In any case, it is highly unlikely
  * that two instances try to access this file. And flock is really
  * causing me grief on my initial steps on Solaris. Some time later,
  * we might re-enable it (or use some alternate method).
  * 2006-02-16 rgerhards
  */

//#if HAVE_FLOCK
  if (flock(fd, LOCK_EX|LOCK_NB) == -1) {
      fscanf(f, "%d", &pid);
      fclose(f);
      printf("Can't lock, lock is held by pid %d.\n", pid);
      return 0;
  }
//#endif

  pid = getpid();
  if (!fprintf(f,"%d\n", pid)) {
      char errStr[1024];
      //rs_strerror_r(errno, errStr, sizeof(errStr));
      printf("Can't write pid , %s.\n", errStr);
      close(fd);
      return 0;
  }
  fflush(f);

//#if HAVE_FLOCK
  if (flock(fd, LOCK_UN) == -1) {
      char errStr[1024];
      //rs_strerror_r(errno, errStr, sizeof(errStr));
      printf("Can't unlock pidfile %s, %s.\n", pidfile, errStr);
      close(fd);
      return 0;
  }
//#endif
  close(fd);

  return pid;
}

/* remove_pid
 *
 * Remove the the specified file. The result from unlink(2)
 * is returned
 */
int remove_pid (char *pidfile)
{
  return unlink (pidfile);
}

